<?php
/**
 * Tenants Management Page
 */
?>

<!-- Enhanced Page Header with Summary Stats -->
<div class="tenants-page-header">
    <div class="header-main">
        <div class="header-content">
            <h1 class="page-heading">
                <i class="fas fa-building"></i>
                Tenant Management
            </h1>
            <p class="page-description">Manage and monitor all tenants in your recommendation engine</p>
        </div>
        <div class="header-actions">
            <button class="btn-outline" onclick="exportData('csv', 'tenants')">
                <i class="fas fa-download"></i>
                <span>Export</span>
            </button>
            <button class="btn-gradient" onclick="openModal('addTenantModal')">
                <i class="fas fa-plus"></i>
                <span>Add Tenant</span>
            </button>
        </div>
    </div>
    
    <!-- Statistics Cards -->
    <div class="stats-overview">
        <div class="stat-card total">
            <div class="stat-icon">
                <i class="fas fa-building"></i>
            </div>
            <div class="stat-content">
                <span class="stat-number"><?php echo format_number($tenant_stats['total']); ?></span>
                <span class="stat-label">Total Tenants</span>
            </div>
        </div>
        
        <div class="stat-card active">
            <div class="stat-icon">
                <i class="fas fa-check-circle"></i>
            </div>
            <div class="stat-content">
                <span class="stat-number"><?php echo format_number($tenant_stats['active']); ?></span>
                <span class="stat-label">Active</span>
            </div>
        </div>
        
        <div class="stat-card inactive">
            <div class="stat-icon">
                <i class="fas fa-pause-circle"></i>
            </div>
            <div class="stat-content">
                <span class="stat-number"><?php echo format_number($tenant_stats['inactive']); ?></span>
                <span class="stat-label">Inactive</span>
            </div>
        </div>
        
        <div class="stat-card suspended">
            <div class="stat-icon">
                <i class="fas fa-ban"></i>
            </div>
            <div class="stat-content">
                <span class="stat-number"><?php echo format_number($tenant_stats['suspended'] ?? 0); ?></span>
                <span class="stat-label">Suspended</span>
            </div>
        </div>
    </div>
</div>

<!-- Enhanced Filters and Search -->
<div class="search-filters-section">
    <div class="search-filters-card">
        <div class="filters-header">
            <h3>
                <i class="fas fa-filter"></i>
                Filter & Search
            </h3>
            <span class="results-count">
                <?php echo format_number($total_tenants); ?> total results
            </span>
        </div>
        
        <form method="GET" class="advanced-filters-form">
            <input type="hidden" name="page" value="tenants">
            
            <div class="filters-row">
                <div class="filter-group">
                    <label class="filter-label">
                        <i class="fas fa-search"></i>
                        Search Tenants
                    </label>
                    <div class="search-input-wrapper">
                        <input type="text" name="search" class="modern-search-input" 
                               placeholder="Search by tenant name or ID..." 
                               value="<?php echo h($_GET['search'] ?? ''); ?>">
                        <i class="fas fa-search search-icon"></i>
                    </div>
                </div>
                
                <div class="filter-group">
                    <label class="filter-label">
                        <i class="fas fa-toggle-on"></i>
                        Status Filter
                    </label>
                    <select name="status" class="modern-select">
                        <option value="">All Status</option>
                        <option value="active" <?php echo ($_GET['status'] ?? '') === 'active' ? 'selected' : ''; ?>>
                            ✓ Active
                        </option>
                        <option value="inactive" <?php echo ($_GET['status'] ?? '') === 'inactive' ? 'selected' : ''; ?>>
                            ⏸ Inactive
                        </option>
                        <option value="suspended" <?php echo ($_GET['status'] ?? '') === 'suspended' ? 'selected' : ''; ?>>
                            🚫 Suspended
                        </option>
                    </select>
                </div>
                
                <div class="filter-actions">
                    <button type="submit" class="btn-search">
                        <i class="fas fa-search"></i>
                        <span>Apply Filters</span>
                    </button>
                    <a href="?page=tenants" class="btn-reset">
                        <i class="fas fa-undo"></i>
                        <span>Reset</span>
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Enhanced Tenants Table -->
<div class="tenants-data-section">
    <div class="modern-table-card">
        <div class="table-card-header">
            <div class="table-title">
                <h3>
                    <i class="fas fa-table"></i>
                    Tenants Overview
                </h3>
                <span class="table-subtitle">Complete list of all tenants and their metrics</span>
            </div>
            <div class="table-meta">
                <span class="showing-count">
                    Showing <strong><?php echo count($tenants); ?></strong> of <strong><?php echo format_number($total_tenants); ?></strong> tenants
                </span>
            </div>
        </div>
        
        <?php if (!empty($tenants)): ?>
            <div class="enhanced-table-container">
                <table class="modern-data-table">
                    <thead>
                        <tr>
                            <th data-sort="tenant_id" class="sortable">
                                <span>Tenant ID</span>
                                <i class="fas fa-sort sort-icon"></i>
                            </th>
                            <th data-sort="name" class="sortable">
                                <span>Tenant Details</span>
                                <i class="fas fa-sort sort-icon"></i>
                            </th>
                            <th data-sort="status" class="sortable">
                                <span>Status</span>
                                <i class="fas fa-sort sort-icon"></i>
                            </th>
                            <th class="text-center">
                                <span>Applications</span>
                            </th>
                            <th class="text-center">
                                <span>Content Items</span>
                            </th>
                            <th class="text-center">
                                <span>Interactions</span>
                                <small>(30 days)</small>
                            </th>
                            <th data-sort="created_at" class="sortable">
                                <span>Created</span>
                                <i class="fas fa-sort sort-icon"></i>
                            </th>
                            <th class="text-center">
                                <span>Actions</span>
                            </th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($tenants as $tenant): ?>
                            <tr class="table-row">
                                <td>
                                    <div class="tenant-id-cell">
                                        <code class="enhanced-tenant-id"><?php echo h($tenant['tenant_id']); ?></code>
                                    </div>
                                </td>
                                <td>
                                    <div class="tenant-details-cell">
                                        <div class="tenant-name-primary"><?php echo h($tenant['name']); ?></div>
                                        <div class="tenant-meta-info">
                                            <span class="tenant-id-small">ID: <?php echo h($tenant['tenant_id']); ?></span>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <div class="status-cell">
                                        <?php echo get_status_badge($tenant['status']); ?>
                                    </div>
                                </td>
                                <td class="text-center">
                                    <div class="metric-cell">
                                        <span class="metric-number"><?php echo format_number($tenant['apps_count']); ?></span>
                                        <span class="metric-label">apps</span>
                                    </div>
                                </td>
                                <td class="text-center">
                                    <div class="metric-cell">
                                        <span class="metric-number"><?php echo format_number($tenant['items_count']); ?></span>
                                        <span class="metric-label">items</span>
                                    </div>
                                </td>
                                <td class="text-center">
                                    <div class="metric-cell interactions-metric">
                                        <span class="metric-number"><?php echo format_number($tenant['interactions_count']); ?></span>
                                        <span class="metric-label">interactions</span>
                                    </div>
                                </td>
                                <td>
                                    <div class="date-cell">
                                        <span class="date-primary" title="<?php echo format_mongo_date($tenant['created_at']); ?>">
                                            <?php echo time_ago($tenant['created_at']); ?>
                                        </span>
                                        <span class="date-secondary">
                                            <?php echo date('M j, Y', strtotime(format_mongo_date($tenant['created_at']))); ?>
                                        </span>
                                    </div>
                                </td>
                                <td>
                                    <div class="actions-cell">
                                        <div class="action-buttons-group">
                                            <button class="action-btn view" title="View Details" 
                                                    onclick="viewTenant('<?php echo h($tenant['tenant_id']); ?>')">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                            <button class="action-btn edit" title="Edit Tenant" 
                                                    onclick="editTenant('<?php echo h($tenant['tenant_id']); ?>')">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <button class="action-btn apps" title="View Applications" 
                                                    onclick="window.location.href='?page=apps&tenant=<?php echo urlencode($tenant['tenant_id']); ?>'">
                                                <i class="fas fa-mobile-alt"></i>
                                            </button>
                                            <?php if ($tenant['apps_count'] == 0 && $tenant['items_count'] == 0): ?>
                                                <button class="action-btn delete" title="Delete Tenant" 
                                                        onclick="deleteTenant('<?php echo h($tenant['tenant_id']); ?>', '<?php echo h($tenant['name']); ?>')">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
                <div class="pagination-section">
                    <?php 
                    $base_url = "?page=tenants" . 
                                (!empty($_GET['search']) ? "&search=" . urlencode($_GET['search']) : "") .
                                (!empty($_GET['status']) ? "&status=" . urlencode($_GET['status']) : "");
                    echo generate_pagination($page_num, $total_pages, $base_url); 
                    ?>
                </div>
            <?php endif; ?>
            
        <?php else: ?>
            <div class="enhanced-empty-state">
                <div class="empty-state-content">
                    <div class="empty-state-icon">
                        <i class="fas fa-building"></i>
                    </div>
                    <h3 class="empty-state-title">No Tenants Found</h3>
                    <p class="empty-state-message">
                        <?php if (!empty($_GET['search']) || !empty($_GET['status'])): ?>
                            No tenants match your current search criteria. Try adjusting your filters.
                        <?php else: ?>
                            Get started by adding your first tenant to the recommendation engine.
                        <?php endif; ?>
                    </p>
                    <div class="empty-state-actions">
                        <?php if (!empty($_GET['search']) || !empty($_GET['status'])): ?>
                            <a href="?page=tenants" class="btn-outline">
                                <i class="fas fa-undo"></i>
                                Clear Filters
                            </a>
                        <?php endif; ?>
                        <button class="btn-gradient" onclick="openModal('addTenantModal')">
                            <i class="fas fa-plus"></i>
                            Add New Tenant
                        </button>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Enhanced Recent Activity -->
<?php if (!empty($recent_tenants)): ?>
<div class="recent-activity-section">
    <div class="activity-overview-card">
        <div class="activity-card-header">
            <div class="activity-title">
                <h3>
                    <i class="fas fa-clock"></i>
                    Recent Activity
                </h3>
                <span class="activity-subtitle">Latest tenant additions and updates</span>
            </div>
            <div class="activity-meta">
                <span class="activity-count"><?php echo count($recent_tenants); ?> recent</span>
            </div>
        </div>
        <div class="enhanced-activity-list">
            <?php foreach ($recent_tenants as $tenant): ?>
                <div class="activity-item">
                    <div class="activity-icon">
                        <i class="fas fa-plus-circle"></i>
                    </div>
                    <div class="activity-details">
                        <div class="activity-main">
                            <span class="activity-action">New tenant added:</span>
                            <strong class="tenant-name-highlight"><?php echo h($tenant['name']); ?></strong>
                        </div>
                        <div class="activity-meta-info">
                            <span class="tenant-id-badge">ID: <?php echo h($tenant['tenant_id']); ?></span>
                            <?php echo get_status_badge($tenant['status']); ?>
                            <span class="activity-timestamp"><?php echo time_ago($tenant['created_at']); ?></span>
                        </div>
                    </div>
                    <div class="activity-actions">
                        <button class="quick-action-btn" title="View Details" 
                                onclick="viewTenant('<?php echo h($tenant['tenant_id']); ?>')">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Add Tenant Modal -->
<div id="addTenantModal" class="modal-overlay" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Add New Tenant</h3>
            <button class="modal-close" onclick="closeModal('addTenantModal')">&times;</button>
        </div>
        <div class="modal-body">
            <form method="POST" data-ajax class="tenant-form">
                <input type="hidden" name="action" value="add">
                
                <div class="form-group">
                    <label for="tenant_name">Tenant Name *</label>
                    <input type="text" id="tenant_name" name="name" required 
                           placeholder="Enter tenant name">
                </div>
                
                <div class="form-group">
                    <label for="tenant_status">Status</label>
                    <select id="tenant_status" name="status">
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                        <option value="suspended">Suspended</option>
                    </select>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn-secondary" onclick="closeModal('addTenantModal')">
                        Cancel
                    </button>
                    <button type="submit" class="btn-primary">
                        <i class="fas fa-plus"></i>
                        Create Tenant
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Tenant Modal -->
<div id="editTenantModal" class="modal-overlay" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Edit Tenant</h3>
            <button class="modal-close" onclick="closeModal('editTenantModal')">&times;</button>
        </div>
        <div class="modal-body">
            <form method="POST" data-ajax class="tenant-form">
                <input type="hidden" name="action" value="update">
                <input type="hidden" id="edit_tenant_id" name="tenant_id">
                
                <div class="form-group">
                    <label for="edit_tenant_name">Tenant Name *</label>
                    <input type="text" id="edit_tenant_name" name="name" required>
                </div>
                
                <div class="form-group">
                    <label for="edit_tenant_status">Status</label>
                    <select id="edit_tenant_status" name="status">
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                        <option value="suspended">Suspended</option>
                    </select>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn-secondary" onclick="closeModal('editTenantModal')">
                        Cancel
                    </button>
                    <button type="submit" class="btn-primary">
                        <i class="fas fa-save"></i>
                        Update Tenant
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Tenant Details Modal -->
<div id="viewTenantModal" class="modal-overlay" style="display: none;">
    <div class="modal-content large">
        <div class="modal-header">
            <h3>Tenant Details</h3>
            <button class="modal-close" onclick="closeModal('viewTenantModal')">&times;</button>
        </div>
        <div class="modal-body">
            <div id="tenantDetails">
                <!-- Content loaded via JavaScript -->
            </div>
        </div>
    </div>
</div>

<style>
/* Enhanced Tenants Page Styles */

/* Page Header Styles */
.tenants-page-header {
    margin-bottom: 32px;
}

.header-main {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 28px;
    background: linear-gradient(135deg, #fff 0%, #f8fafc 100%);
    padding: 32px;
    border-radius: 16px;
    border: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
}

.header-content {
    flex: 1;
}

.page-heading {
    font-size: 1.875rem;
    font-weight: 800;
    color: #1e293b;
    margin-bottom: 8px;
    display: flex;
    align-items: center;
    gap: 16px;
}

.page-heading i {
    color: #dc2626;
    font-size: 1.875rem;
}

.page-description {
    font-size: 1rem;
    color: #64748b;
    font-weight: 400;
    line-height: 1.5;
}

.header-actions {
    display: flex;
    gap: 12px;
    align-items: center;
}

.btn-outline {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    background: white;
    border: 2px solid #e2e8f0;
    border-radius: 10px;
    color: #475569;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.3s ease;
    cursor: pointer;
}

.btn-outline:hover {
    border-color: #dc2626;
    color: #dc2626;
    background: #fef2f2;
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(220, 38, 38, 0.15);
}

.btn-gradient {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: linear-gradient(135deg, #dc2626 0%, #f97316 100%);
    border: none;
    border-radius: 10px;
    color: white;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 4px 12px rgba(220, 38, 38, 0.3);
}

.btn-gradient:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 20px rgba(220, 38, 38, 0.4);
}

/* Statistics Overview */
.stats-overview {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
    gap: 20px;
}

.stat-card {
    background: white;
    border-radius: 16px;
    padding: 24px;
    border: 1px solid #e2e8f0;
    display: flex;
    align-items: center;
    gap: 20px;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 3px;
    background: linear-gradient(90deg, transparent, rgba(220, 38, 38, 0.5), transparent);
}

.stat-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 12px 24px rgba(0, 0, 0, 0.1);
}

.stat-card.total::before { background: linear-gradient(90deg, #1e293b, #475569); }
.stat-card.active::before { background: linear-gradient(90deg, #059669, #10b981); }
.stat-card.inactive::before { background: linear-gradient(90deg, #dc2626, #f87171); }
.stat-card.suspended::before { background: linear-gradient(90deg, #f59e0b, #fbbf24); }

.stat-icon {
    width: 56px;
    height: 56px;
    border-radius: 14px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 9px;
    color: white;
    flex-shrink: 0;
}

.stat-card.total .stat-icon { background: linear-gradient(135deg, #1e293b, #475569); }
.stat-card.active .stat-icon { background: linear-gradient(135deg, #059669, #10b981); }
.stat-card.inactive .stat-icon { background: linear-gradient(135deg, #dc2626, #f87171); }
.stat-card.suspended .stat-icon { background: linear-gradient(135deg, #f59e0b, #fbbf24); }

.stat-content {
    flex: 1;
}

.stat-number {
    display: block;
    font-size: 1.5rem;
    font-weight: 800;
    color: #1e293b;
    line-height: 1;
    margin-bottom: 4px;
}

.stat-label {
    font-size: 0.875rem;
    color: #64748b;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.025em;
}

/* Search and Filters */
.search-filters-section {
    margin-bottom: 32px;
}

.search-filters-card {
    background: white;
    border-radius: 16px;
    border: 1px solid #e2e8f0;
    padding: 24px;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
}

.filters-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 24px;
    padding-bottom: 16px;
    border-bottom: 1px solid #f1f5f9;
}

.filters-header h3 {
    font-size: 1.125rem;
    font-weight: 700;
    color: #1e293b;
    display: flex;
    align-items: center;
    gap: 10px;
}

.filters-header i {
    color: #dc2626;
}

.results-count {
    font-size: 0.875rem;
    color: #64748b;
    font-weight: 500;
}

.advanced-filters-form {
    width: 100%;
}

.filters-row {
    display: grid;
    grid-template-columns: 2fr 1fr auto;
    gap: 24px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.filter-label {
    font-size: 0.875rem;
    font-weight: 600;
    color: #374151;
    display: flex;
    align-items: center;
    gap: 8px;
}

.filter-label i {
    color: #94a3b8;
    font-size: 0.875rem;
}

.search-input-wrapper {
    position: relative;
}

.modern-search-input {
    width: 100%;
    padding: 14px 16px 14px 44px;
    border: 2px solid #e2e8f0;
    border-radius: 12px;
    font-size: 1rem;
    background: #f8fafc;
    transition: all 0.3s ease;
    color: #1e293b;
}

.modern-search-input:focus {
    outline: none;
    border-color: #dc2626;
    background: white;
    box-shadow: 0 0 0 3px rgba(220, 38, 38, 0.1);
}

.search-icon {
    position: absolute;
    left: 16px;
    top: 50%;
    transform: translateY(-50%);
    color: #94a3b8;
    font-size: 1rem;
}

.modern-select {
    width: 100%;
    padding: 14px 16px;
    border: 2px solid #e2e8f0;
    border-radius: 12px;
    font-size: 1rem;
    background: #f8fafc;
    color: #1e293b;
    transition: all 0.3s ease;
    cursor: pointer;
}

.modern-select:focus {
    outline: none;
    border-color: #dc2626;
    background: white;
    box-shadow: 0 0 0 3px rgba(220, 38, 38, 0.1);
}

.filter-actions {
    display: flex;
    gap: 12px;
}

.btn-search, .btn-reset {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 14px 20px;
    border-radius: 12px;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.3s ease;
    cursor: pointer;
    white-space: nowrap;
}

.btn-search {
    background: linear-gradient(135deg, #dc2626, #f97316);
    border: none;
    color: white;
    box-shadow: 0 4px 12px rgba(220, 38, 38, 0.3);
}

.btn-search:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 16px rgba(220, 38, 38, 0.4);
}

.btn-reset {
    background: white;
    border: 2px solid #e2e8f0;
    color: #64748b;
}

.btn-reset:hover {
    border-color: #64748b;
    color: #374151;
    background: #f8fafc;
}

/* Enhanced Table Styles */
.tenants-data-section {
    margin-bottom: 32px;
}

.modern-table-card {
    background: white;
    border-radius: 16px;
    border: 1px solid #e2e8f0;
    overflow: hidden;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
}

.table-card-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    padding: 24px 28px;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-bottom: 1px solid #e2e8f0;
}

.table-title h3 {
    font-size: 1.125rem;
    font-weight: 700;
    color: #1e293b;
    margin-bottom: 4px;
    display: flex;
    align-items: center;
    gap: 12px;
}

.table-title i {
    color: #dc2626;
    font-size: 1.125rem;
}

.table-subtitle {
    font-size: 0.875rem;
    color: #64748b;
    font-weight: 400;
}

.showing-count {
    font-size: 0.875rem;
    color: #64748b;
    font-weight: 500;
}

.showing-count strong {
    color: #1e293b;
    font-weight: 700;
}

.enhanced-table-container {
    overflow-x: auto;
}

.modern-data-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 1rem;
}

.modern-data-table thead {
    background: #f8fafc;
}

.modern-data-table th {
    padding: 18px 20px;
    text-align: left;
    font-weight: 600;
    color: #475569;
    font-size: 0.875rem;
    text-transform: uppercase;
    letter-spacing: 0.025em;
    border-bottom: 2px solid #e2e8f0;
    position: relative;
}

.modern-data-table th.sortable {
    cursor: pointer;
    user-select: none;
    transition: all 0.2s ease;
}

.modern-data-table th.sortable:hover {
    background: #f1f5f9;
    color: #dc2626;
}

.modern-data-table th.text-center {
    text-align: center;
}

.sort-icon {
    margin-left: 6px;
    color: #94a3b8;
    font-size: 0.875rem;
}

.modern-data-table th small {
    display: block;
    font-size: 0.75rem;
    font-weight: 400;
    color: #94a3b8;
    text-transform: none;
    margin-top: 2px;
}

.table-row {
    border-bottom: 1px solid #f1f5f9;
    transition: all 0.2s ease;
}

.table-row:hover {
    background: #f8fafc;
}

.modern-data-table td {
    padding: 20px;
    vertical-align: middle;
}

.text-center {
    text-align: center;
}

/* Table Cell Styles */
.tenant-id-cell {
    display: flex;
    align-items: center;
}

.enhanced-tenant-id {
    background: linear-gradient(135deg, #1e293b, #475569);
    color: white;
    padding: 6px 12px;
    border-radius: 8px;
    font-family: 'Inter', 'Lato', 'Monaco', 'Menlo', monospace;
    font-size: 0.875rem;
    font-weight: 600;
    letter-spacing: 0.5px;
}

.tenant-details-cell {
    max-width: 200px;
}

.tenant-name-primary {
    font-size: 1rem;
    font-weight: 700;
    color: #1e293b;
    margin-bottom: 4px;
    line-height: 1.2;
}

.tenant-meta-info {
    display: flex;
    align-items: center;
    gap: 8px;
}

.tenant-id-small {
    font-size: 0.75rem;
    color: #64748b;
    font-family: 'Inter', 'Lato', monospace;
}

.status-cell {
    display: flex;
    justify-content: flex-start;
    align-items: center;
}

.metric-cell {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 2px;
}

.metric-number {
    font-size: 1.125rem;
    font-weight: 800;
    color: #1e293b;
    line-height: 1;
}

.metric-label {
    font-size: 0.75rem;
    color: #94a3b8;
    text-transform: uppercase;
    letter-spacing: 0.025em;
    font-weight: 500;
}

.interactions-metric .metric-number {
    color: #f59e0b;
}

.date-cell {
    display: flex;
    flex-direction: column;
    gap: 2px;
}

.date-primary {
    font-size: 0.875rem;
    font-weight: 600;
    color: #1e293b;
}

.date-secondary {
    font-size: 0.75rem;
    color: #94a3b8;
}

.actions-cell {
    display: flex;
    justify-content: center;
}

.action-buttons-group {
    display: flex;
    gap: 6px;
}

.action-btn {
    width: 36px;
    height: 36px;
    border-radius: 8px;
    border: none;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.875rem;
}

.action-btn.view {
    background: #f0f9ff;
    color: #0369a1;
}

.action-btn.view:hover {
    background: #0369a1;
    color: white;
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(3, 105, 161, 0.3);
}

.action-btn.edit {
    background: #f0fdf4;
    color: #059669;
}

.action-btn.edit:hover {
    background: #059669;
    color: white;
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(5, 150, 105, 0.3);
}

.action-btn.apps {
    background: #fef3c7;
    color: #f59e0b;
}

.action-btn.apps:hover {
    background: #f59e0b;
    color: white;
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(245, 158, 11, 0.3);
}

.action-btn.delete {
    background: #fef2f2;
    color: #dc2626;
}

.action-btn.delete:hover {
    background: #dc2626;
    color: white;
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(220, 38, 38, 0.3);
}

/* Enhanced Empty State */
.enhanced-empty-state {
    padding: 80px 40px;
    text-align: center;
}

.empty-state-content {
    max-width: 400px;
    margin: 0 auto;
}

.empty-state-icon {
    width: 80px;
    height: 80px;
    margin: 0 auto 24px;
    background: linear-gradient(135deg, #f1f5f9, #e2e8f0);
    border-radius: 20px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    color: #94a3b8;
}

.empty-state-title {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1e293b;
    margin-bottom: 12px;
}

.empty-state-message {
    font-size: 1rem;
    color: #64748b;
    line-height: 1.6;
    margin-bottom: 32px;
}

.empty-state-actions {
    display: flex;
    gap: 12px;
    justify-content: center;
    flex-wrap: wrap;
}

/* Enhanced Activity Section */
.recent-activity-section {
    margin-bottom: 32px;
}

.activity-overview-card {
    background: white;
    border-radius: 16px;
    border: 1px solid #e2e8f0;
    overflow: hidden;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
}

.activity-card-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    padding: 24px 28px;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-bottom: 1px solid #e2e8f0;
}

.activity-title h3 {
    font-size: 1.125rem;
    font-weight: 700;
    color: #1e293b;
    margin-bottom: 4px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.activity-title i {
    color: #f59e0b;
}

.activity-subtitle {
    font-size: 0.875rem;
    color: #64748b;
    font-weight: 400;
}

.activity-count {
    font-size: 0.75rem;
    color: #94a3b8;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.025em;
}

.enhanced-activity-list {
    padding: 20px 28px;
}

.activity-item {
    display: flex;
    align-items: center;
    gap: 16px;
    padding: 16px 0;
    border-bottom: 1px solid #f1f5f9;
}

.activity-item:last-child {
    border-bottom: none;
}

.activity-icon {
    width: 40px;
    height: 40px;
    border-radius: 10px;
    background: linear-gradient(135deg, #10b981, #059669);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1rem;
    flex-shrink: 0;
}

.activity-details {
    flex: 1;
}

.activity-main {
    margin-bottom: 6px;
}

.activity-action {
    font-size: 0.875rem;
    color: #64748b;
    margin-right: 6px;
}

.tenant-name-highlight {
    font-size: 0.875rem;
    font-weight: 700;
    color: #1e293b;
}

.activity-meta-info {
    display: flex;
    align-items: center;
    gap: 12px;
}

.tenant-id-badge {
    font-size: 0.75rem;
    background: #f1f5f9;
    color: #64748b;
    padding: 2px 6px;
    border-radius: 4px;
    font-family: 'Inter', 'Lato', monospace;
}

.activity-timestamp {
    font-size: 0.75rem;
    color: #94a3b8;
}

.quick-action-btn {
    width: 32px;
    height: 32px;
    border-radius: 8px;
    border: none;
    background: #f8fafc;
    color: #64748b;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s ease;
}

.quick-action-btn:hover {
    background: #dc2626;
    color: white;
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(220, 38, 38, 0.3);
}

/* Responsive Design */
@media (max-width: 1200px) {
    .filters-row {
        grid-template-columns: 1fr;
        gap: 16px;
    }
    
    .filter-actions {
        justify-content: flex-start;
    }
}

@media (max-width: 768px) {
    .header-main {
        flex-direction: column;
        gap: 20px;
    }
    
    .header-actions {
        width: 100%;
        justify-content: flex-start;
    }
    
    .stats-overview {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .enhanced-table-container {
        font-size: 0.875rem;
    }
    
    .tenant-name-primary {
        font-size: 0.875rem;
    }
    
    .metric-number {
        font-size: 1rem;
    }
}

@media (max-width: 640px) {
    .stats-overview {
        grid-template-columns: 1fr;
    }
    
    .action-buttons-group {
        flex-direction: column;
        gap: 4px;
    }
    
    .action-btn {
        width: 32px;
        height: 32px;
    }
}
</style>

<script>
// Tenant management functions
function viewTenant(tenantId) {
    showLoading();
    
    fetch(`api/tenant-details.php?tenant_id=${tenantId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                document.getElementById('tenantDetails').innerHTML = data.html;
                openModal('viewTenantModal');
            } else {
                showToast(data.message || 'Failed to load tenant details', 'error');
            }
        })
        .catch(error => {
            showToast('Error loading tenant details', 'error');
        })
        .finally(() => {
            hideLoading();
        });
}

function editTenant(tenantId) {
    showLoading();
    
    fetch(`api/get-tenant.php?tenant_id=${tenantId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const tenant = data.tenant;
                document.getElementById('edit_tenant_id').value = tenant.tenant_id;
                document.getElementById('edit_tenant_name').value = tenant.name;
                document.getElementById('edit_tenant_status').value = tenant.status;
                openModal('editTenantModal');
            } else {
                showToast(data.message || 'Failed to load tenant data', 'error');
            }
        })
        .catch(error => {
            showToast('Error loading tenant data', 'error');
        })
        .finally(() => {
            hideLoading();
        });
}

function deleteTenant(tenantId, tenantName) {
    if (confirm(`Are you sure you want to delete tenant "${tenantName}"? This action cannot be undone.`)) {
        showLoading();
        
        const formData = new FormData();
        formData.append('action', 'delete');
        formData.append('tenant_id', tenantId);
        
        fetch('?page=tenants', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast(data.message, 'success');
                window.location.reload();
            } else {
                showToast(data.message || 'Failed to delete tenant', 'error');
            }
        })
        .catch(error => {
            showToast('Error deleting tenant', 'error');
        })
        .finally(() => {
            hideLoading();
        });
    }
}

// Initialize page
document.addEventListener('DOMContentLoaded', function() {
    // Auto-submit filter form on select change
    document.querySelectorAll('.filter-select').forEach(select => {
        select.addEventListener('change', function() {
            this.closest('form').submit();
        });
    });
});
</script>
